import requests     # To send post/get packets to Spaceship API
import json         # To parse configuration data
import time         # To delay the time between DNS checks
import socket       # To check if there is an active internet connection

from DDNS import Loader


class DDNS:
    def __init__(self):
        self.loader = Loader.Load()
        self.config = self.loader.get_config()

# Config
        self.domain                  = self.config["domain"]
        self.apiKey                  = self.config["api_key"]
        self.apiSecret               = self.config["api_secret"]
        self.checkIntervalSeconds    = self.config["check_interval_seconds"]
        self.dnsRecord               = self.config["dns_record"]
        self.ipLookup                = self.config["ip_lookup"]
        self.localIpFile             = self.config["local_ip_file"]

# Request
        self.API_URL = f"https://spaceship.dev/api/v1/dns/records/{self.domain}"
        self.HEADERS = {
            "X-API-Key": self.apiKey,
            "X-API-Secret": self.apiSecret,
            "content-type": "application/json"
        }

# DNS Record
        self.DNSRecord = {}
# Public IP
        self.PublicIP = ""




    # Function to check if there is an active internet connection
    def has_internet_connection(self):
        socket.setdefaulttimeout(10)
        # --- Information --- #
        # Host:     8.8.8.8 (google-public-dns-a.google.com)
        # Port:     53 (Open TCP port)
        # Service:  Domain (DNS/TCP)
        # --- End Information --- #
        try:
            # Attempt to connect to 8.8.8.8
            socket.setdefaulttimeout(19) # Timeout period
            socket.socket(socket.AF_INET, socket.SOCK_STREAM).connect(("8.8.8.8", 53))
            # Connection successful
            return True
        except socket.error:
            # Connection unsuccessful
            return False


# Get public ip address from ip_lookup
    def get_public_address(self):
        req = requests.get(self.ipLookup)
        if req.status_code == 200:
            self.PublicIP = req.text
        return None


# Get DNS record from API
    def get_DNS_record(self):
        req = requests.get(self.API_URL+"?take=50&skip=0", headers=self.HEADERS)

        if req.status_code == 200:
            records = req.json().get("items", [])
            # Iterate through the list of records to find the one with the matching name
            for record in records:
                if record.get('name') == self.dnsRecord:
                    self.DNSRecord = record
                    return True
            # Return None if the record isn't found
            return None
        else:
            print(f"[ERROR] Get: records {req.status_code}: {req.text}")
            return None


# Check weather record is up to date
    def check_record(self):

        # Get current public IP address
        self.get_public_address()

        # Get current DNS record
        self.get_DNS_record()

        if self.PublicIP and self.DNSRecord:
            if self.DNSRecord.get("address") != self.PublicIP:
                print(f"[INFO] {self.dnsRecord} is outdated: {self.DNSRecord.get('address')} != {self.PublicIP}")
                # Update DNS record
                self.update_record()
                return False
            print(f"[INFO] {self.dnsRecord} is up to date.")
            return True
        return False


# Delete old DNS record
    def delete_old_record(self):

        # Get current DNS record address
        address = self.DNSRecord.get("address")

        # Payload to delete record
        payload = [
            {
                "type": "A",
                "name": self.dnsRecord,
                "address": address
            }
        ]

        # Delete old DNS record
        req = requests.delete(self.API_URL, json=payload, headers=self.HEADERS)

        if req.status_code == 204:
            return True
        else:
            return False


# Create updated DNS record
    def create_updated_record(self):

        payload = {
            "force": True,
            "items": [
                {
                    "type": "A",
                    "name": self.dnsRecord,
                    "address": self.PublicIP,
                    "ttl": self.DNSRecord.get("ttl", 3600),
                }
            ]
        }

        # Create new updated DNS record
        req = requests.put(self.API_URL, headers=self.HEADERS, json=payload)
        
        if req.status_code == 200 or req.status_code == 204:
            return True
        else:
            return False


# Function to update DNS record
    def update_record(self):

        if self.PublicIP and self.DNSRecord:
            print(f"[INFO] Updating {self.dnsRecord} to {self.PublicIP}")

    # Delete old DNS record
            if self.delete_old_record():
                print(f"[INFO] Successfully deleted old DNS record {self.dnsRecord}")
            else:
                print(f"[ERROR] Failed to delete old DNS record {self.dnsRecord}")
                return False

    # Create updated DNS record
            if self.create_updated_record():
                print(f"[INFO] Successfully created updated DNS record {self.dnsRecord}")
            else:
                print(f"[ERROR] Failed to create updated DNS record {self.dnsRecord}")
                return False

            print(f"[INFO] Successfully Updated {self.dnsRecord} to {self.PublicIP}")
            return True

        return False




if __name__ == "__main__":
    ddns = DDNS()
    while True:
        if not ddns.has_internet_connection():
            print("[WAITING] No internet connection.")
            time.sleep(10)
            continue

        ddns.check_record()
        time.sleep(ddns.checkIntervalSeconds)